using System;
using System.Collections;
using System.Data;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>3/9/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Holds collection of VBECS user's division roles.</summary>

	#endregion

	public class VbecsUserDivisionRolesCollection : IEnumerable
	{
		private const int TypicalNumberOfDivisions = 5;

		private Hashtable _divisionRoles;
		private DivisionCollection _allUserDivisions;
		private DivisionCollection _effectiveDivisions;

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6940"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>New empty instance of the class. </ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6941"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Blank constructor with variable initialization
		/// </summary>
		public VbecsUserDivisionRolesCollection()
		{
			_divisionRoles = new Hashtable( TypicalNumberOfDivisions );
			_allUserDivisions = new DivisionCollection();
			_effectiveDivisions = new DivisionCollection();			
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6868"> 
		///		<ExpectedInput>Non-empty DataTable containing a list of division roles.</ExpectedInput>
		///		<ExpectedOutput>Non-empty collection.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6869"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///		Constructs a collection of <see cref="VbecsUserDivisionRole"/> objects from the supplied <see cref="DataTable"/>.
		///		Used to deserialize user divisional access information from the DB. 
		/// </summary>
		/// <param name="sourceDataTable">Source <see cref="DataTable"/> containing user divisional access data.</param>
		public VbecsUserDivisionRolesCollection( DataTable sourceDataTable ) 
			: this()
		{
			if( sourceDataTable == null )
				throw( new ArgumentNullException( "sourceDataTable" ) );

			foreach( DataRow dr in sourceDataTable.Rows )
				this.Add( new VbecsUserDivisionRole( dr ) );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6866"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6867"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6942"> 
		///		<ExpectedInput>Division role that is already in the collection.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds a given <see cref="VbecsUserDivisionRole"/> to the collection.
		/// </summary>
		/// <param name="roleToAdd">Role to add to the collection.</param>
		/// <returns>Reference to the added role.</returns>
		public VbecsUserDivisionRole Add( VbecsUserDivisionRole roleToAdd )
		{
			if( roleToAdd == null )
				throw( new ArgumentNullException( "roleToAdd" ) );

			if( _divisionRoles.ContainsKey( roleToAdd.Division.DivisionCode.Trim() ) )
				throw( new ArgumentException( Common.StrRes.SysErrMsg.Common.VbecsUserRoleIsAlreadyInTheCollection( roleToAdd.Division.DivisionCode ).ResString ) );
			
			_divisionRoles.Add( roleToAdd.Division.DivisionCode.Trim(), roleToAdd );
			_allUserDivisions.Add( roleToAdd.Division );

			if( roleToAdd.IsActive && roleToAdd.Division.IsActive )
				_effectiveDivisions.Add( roleToAdd.Division );

			roleToAdd.RoleChanged += new EventHandler( OnRoleInCollectionChanged );

			return roleToAdd;
		}

		/// <summary>
		/// Triggers notifications whenever collection changes. 
		/// </summary>
		/// <param name="sender">Sender object.</param>
		/// <param name="e">Event arguments.</param>
		protected void OnRoleInCollectionChanged( object sender, EventArgs e )
		{
			UpdateEffectiveDivisionListOnRoleChange( ((VbecsUserDivisionRole)sender) );
		}

		/// <summary>
		/// Adds or removes division corresponding to role to/from effective divisions list. 
		/// </summary>
		protected void UpdateEffectiveDivisionListOnRoleChange( VbecsUserDivisionRole changedRole )
		{
			if( changedRole == null )
				throw( new ArgumentNullException( "changedRole" ) );

			int indexInEffectiveDivsList = _effectiveDivisions.IndexOf( changedRole.Division ); 

			if( ( changedRole.IsActive && changedRole.Division.IsActive ) == ( indexInEffectiveDivsList != -1 ) ) // role presence in the effective list corresponds to its status
				return;

			if( changedRole.IsActive )
				_effectiveDivisions.Add( changedRole.Division );
			else
				_effectiveDivisions.RemoveAt( indexInEffectiveDivsList );
		}

		/// <summary>
		/// Exports list of divisions and roles into save table for a given user. 
		/// </summary>
		internal DataTable ExportToSaveDataTable( VbecsUser userToSaveRolesFor )
		{
			if( userToSaveRolesFor == null )
				throw( new ArgumentNullException( "userToSaveRolesFor" ) );

			DataTable dt = DAL.VbecsUser.GetVbecsUserDivisionRoleSaveTemplateDataTable();

			foreach( VbecsUserDivisionRole role in _divisionRoles.Values )
				if( role.IsDirty )
					dt.Rows.Add( role.LoadDataRowFromThis( dt.NewRow(), userToSaveRolesFor ) );	

			return dt;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6862"> 
		///		<ExpectedInput>Valid division code.</ExpectedInput>
		///		<ExpectedOutput>User role corresponding for the division with the specified code.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6863"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// An indexer providing an access to <see cref="VbecsUserDivisionRole"/> 
		/// in the collection by division code. 
		/// </summary>
		public VbecsUserDivisionRole this[ string divisionCode ] 
		{
			get
			{
				if( divisionCode == null )
					throw( new ArgumentNullException( "divisionCode" ) );

				return (VbecsUserDivisionRole)_divisionRoles[ divisionCode.Trim() ];
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6860"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6861"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// All divisions (including inactive) that are listed for 
		/// the user (including the ones where user role was inactivated).
		/// </summary>
		public DivisionCollection AllDivisions
		{
			get
			{
				return _allUserDivisions;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6856"> 
		///		<ExpectedInput>VbecsUserDivisionRolesCollection defined for the valid VBECS user.</ExpectedInput>
		///		<ExpectedOutput>Non-null, non-empty DivisionCollection.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6857"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// List of divisions that user can access 
		/// (active divisions where user has active roles). 
		/// </summary>
		public DivisionCollection EffectiveDivisions
		{
			get
			{
				return _effectiveDivisions;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6858"> 
		///		<ExpectedInput>VbecsUserDivisionRolesCollection defined for the valid VBECS user.</ExpectedInput>
		///		<ExpectedOutput>Positive integer.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6859"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of items in the collection. 
		/// </summary>
		public int Count
		{
			get
			{
				return _divisionRoles.Count;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6938"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-null enumerator positioned before the first element in the collection.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6939"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Returns an enumerator that can iterate through a collection.
		/// </summary>
		/// <returns>An <see cref="IEnumerator"/> that can be used to iterate through the collection.</returns>
		public IEnumerator GetEnumerator()
		{			
			return _divisionRoles.Values.GetEnumerator();
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8038"> 
		///		<ExpectedInput>Collection containing dirty division role</ExpectedInput>
		///		<ExpectedOutput>IsDirty property = true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8039"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsDirty
		/// </summary>
		public bool IsDirty
		{
			get
			{
				foreach( VbecsUserDivisionRole role in _divisionRoles.Values )
					if( role.IsDirty )
						return true;

				return false;
			}
		}
	}
}
